<?php

namespace App\Filament\Resources;

use App\Filament\Resources\LicenseResource\Pages;
use App\Models\License;
use App\Models\LicenseActivationLog;
use App\Services\LicenseKeyGenerator;
use Filament\Forms;
use Filament\Forms\Components\DateTimePicker;
use Filament\Forms\Components\Section;
use Filament\Forms\Components\Select;
use Filament\Forms\Components\Textarea;
use Filament\Forms\Components\TextInput;
use Filament\Forms\Components\Toggle;
use Filament\Forms\Form;
use Filament\Forms\Get;
use Filament\Notifications\Notification;
use Filament\Resources\Resource;
use Filament\Tables;
use Filament\Tables\Actions\Action;
use Filament\Tables\Actions\ActionGroup;
use Filament\Tables\Columns\BadgeColumn;
use Filament\Tables\Columns\TextColumn;
use Filament\Tables\Filters\SelectFilter;
use Filament\Tables\Filters\TrashedFilter;
use Filament\Tables\Table;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\SoftDeletingScope;

class LicenseResource extends Resource
{
    protected static ?string $model = License::class;

    protected static ?string $navigationIcon  = 'heroicon-o-key';
    protected static ?string $navigationLabel = 'Lisanslar';
    protected static ?string $modelLabel      = 'Lisans';
    protected static ?string $pluralModelLabel = 'Lisanslar';
    protected static ?string $navigationGroup = 'Lisans Yönetimi';
    protected static ?int    $navigationSort  = 1;

    // Global arama alanları
    protected static ?string $recordTitleAttribute = 'license_key';

    public static function getGloballySearchableAttributes(): array
    {
        return ['license_key', 'customer_name', 'customer_email', 'domain'];
    }

    // =========================================================================
    // FORM
    // =========================================================================

    public static function form(Form $form): Form
    {
        /** @var LicenseKeyGenerator $generator */
        $generator = app(LicenseKeyGenerator::class);

        return $form->schema([

            // ── Bölüm 1: Lisans Ana Bilgileri ──────────────────────────────────
            Section::make('Lisans Ana Bilgileri')
                ->description('Lisans anahtarını ve kullanım limitlerini buradan yönetin.')
                ->icon('heroicon-o-key')
                ->schema([

                    TextInput::make('license_key')
                        ->label('Lisans Anahtarı')
                        ->default(fn () => $generator->generate())
                        ->required()
                        ->unique(ignoreRecord: true)
                        ->maxLength(19)
                        ->placeholder('XXXX-XXXX-XXXX-XXXX')
                        ->fontFamily('mono')
                        ->suffixAction(
                            Forms\Components\Actions\Action::make('generate')
                                ->label('Yeni Üret')
                                ->icon('heroicon-m-arrow-path')
                                ->color('gray')
                                ->action(function (Forms\Set $set) use ($generator) {
                                    $set('license_key', $generator->generateUnique());
                                })
                        )
                        ->helperText('Format: XXXX-XXXX-XXXX-XXXX — "Yeni Üret" ile otomatik oluşturabilirsiniz.'),

                    Select::make('status')
                        ->label('Durum')
                        ->options([
                            'active'    => 'Aktif',
                            'suspended' => 'Askıya Alındı',
                            'expired'   => 'Süresi Doldu',
                        ])
                        ->default('active')
                        ->required()
                        ->native(false)
                        ->suffixIcon('heroicon-m-check-badge'),

                    TextInput::make('activation_limit')
                        ->label('Aktivasyon Limiti')
                        ->numeric()
                        ->default(1)
                        ->minValue(1)
                        ->maxValue(100)
                        ->required()
                        ->helperText('Kaç farklı domain\'de kullanılabilir?'),

                    TextInput::make('current_activations')
                        ->label('Mevcut Aktivasyon')
                        ->numeric()
                        ->default(0)
                        ->disabled()
                        ->dehydrated(false)
                        ->helperText('Otomatik güncellenir.'),

                ])->columns(2),

            // ── Bölüm 2: Müşteri Bilgileri ─────────────────────────────────────
            Section::make('Müşteri Bilgileri')
                ->description('Lisansın sahibi müşteriye ait bilgiler.')
                ->icon('heroicon-o-user')
                ->schema([

                    TextInput::make('customer_name')
                        ->label('Müşteri Adı Soyadı')
                        ->required()
                        ->maxLength(150)
                        ->placeholder('Ahmet Yılmaz'),

                    TextInput::make('customer_email')
                        ->label('E-posta Adresi')
                        ->email()
                        ->required()
                        ->maxLength(255)
                        ->placeholder('ahmet@example.com'),

                    TextInput::make('domain')
                        ->label('Yetkili Domain')
                        ->maxLength(255)
                        ->placeholder('orneksite.com')
                        ->helperText('Aktivasyon sonrası otomatik kilitlenir. Değiştirmek deaktivasyon gerektirir.'),

                ])->columns(2),

            // ── Bölüm 3: Süre Yönetimi ─────────────────────────────────────────
            Section::make('Süre Yönetimi')
                ->description('Lisansın geçerlilik süresini belirleyin.')
                ->icon('heroicon-o-calendar-days')
                ->schema([

                    Toggle::make('is_unlimited')
                        ->label('Sınırsız Lisans')
                        ->helperText('Açıksa bitiş tarihi uygulanmaz.')
                        ->default(true)
                        ->live()
                        ->afterStateUpdated(function (Forms\Set $set, $state) {
                            if ($state) {
                                $set('valid_until', null);
                            }
                        })
                        ->columnSpanFull(),

                    DateTimePicker::make('valid_until')
                        ->label('Bitiş Tarihi')
                        ->hidden(fn (Get $get) => (bool) $get('is_unlimited'))
                        ->required(fn (Get $get) => !(bool) $get('is_unlimited'))
                        ->minDate(now())
                        ->native(false)
                        ->displayFormat('d.m.Y H:i')
                        ->timezone('Europe/Istanbul')
                        ->helperText('Lisansın sona ereceği tarih ve saat.')
                        ->columnSpanFull(),

                ])->columns(1),

            // ── Bölüm 4: Uzaktan Yönetim ───────────────────────────────────────
            Section::make('Uzaktan Yönetim')
                ->description('Bu mesaj müşterinin WordPress yönetici panelinde anlık görünür.')
                ->icon('heroicon-o-chat-bubble-left-ellipsis')
                ->schema([

                    Textarea::make('admin_message')
                        ->label('Eklenti Paneli Mesajı')
                        ->placeholder('Müşterinin WP panelinde görünecek özel notu buraya yazın... Örn: "Lisansınız 3 gün içinde sona eriyor, lütfen yenileyin."')
                        ->rows(3)
                        ->maxLength(1000)
                        ->columnSpanFull(),

                ]),

        ]);
    }

    // =========================================================================
    // TABLE
    // =========================================================================

    public static function table(Table $table): Table
    {
        return $table
            ->columns([

                TextColumn::make('license_key')
                    ->label('Lisans Anahtarı')
                    ->fontFamily('mono')
                    ->searchable()
                    ->sortable()
                    ->copyable()
                    ->copyMessage('Lisans anahtarı kopyalandı!')
                    ->weight('bold'),

                TextColumn::make('customer_name')
                    ->label('Müşteri')
                    ->searchable()
                    ->sortable()
                    ->description(fn (License $record) => $record->customer_email),

                TextColumn::make('domain')
                    ->label('Domain')
                    ->icon('heroicon-m-globe-alt')
                    ->searchable()
                    ->placeholder('Henüz aktive edilmedi')
                    ->color(fn (License $record) => $record->domain ? 'success' : 'gray'),

                // ── Durum Rozeti ──────────────────────────────────────────────
                BadgeColumn::make('status')
                    ->label('Durum')
                    ->formatStateUsing(fn (string $state) => match ($state) {
                        'active'    => 'Aktif',
                        'suspended' => 'Askıya Alındı',
                        'expired'   => 'Süresi Doldu',
                        default     => $state,
                    })
                    ->color(fn (string $state) => match ($state) {
                        'active'    => 'success',   // Yeşil
                        'suspended' => 'danger',    // Kırmızı
                        'expired'   => 'warning',   // Turuncu
                        default     => 'gray',
                    })
                    ->icon(fn (string $state) => match ($state) {
                        'active'    => 'heroicon-m-check-circle',
                        'suspended' => 'heroicon-m-no-symbol',
                        'expired'   => 'heroicon-m-clock',
                        default     => null,
                    })
                    ->sortable(),

                TextColumn::make('current_activations')
                    ->label('Aktiv.')
                    ->formatStateUsing(
                        fn (License $record) =>
                            "{$record->current_activations} / {$record->activation_limit}"
                    )
                    ->color(fn (License $record) =>
                        $record->current_activations >= $record->activation_limit
                            ? 'danger'
                            : 'success'
                    )
                    ->alignCenter(),

                TextColumn::make('valid_until')
                    ->label('Bitiş Tarihi')
                    ->dateTime('d.m.Y')
                    ->timezone('Europe/Istanbul')
                    ->placeholder('Sınırsız')
                    ->color(fn (License $record) => match (true) {
                        $record->valid_until === null                             => 'gray',
                        $record->valid_until->isPast()                           => 'danger',
                        $record->valid_until->lt(now()->addDays(7))              => 'warning',
                        default                                                  => 'success',
                    })
                    ->sortable(),

                TextColumn::make('last_verify_at')
                    ->label('Son Doğrulama')
                    ->dateTime('d.m.Y H:i')
                    ->timezone('Europe/Istanbul')
                    ->placeholder('Henüz doğrulanmadı')
                    ->since()
                    ->sortable(),

            ])

            // ── Filtreler ─────────────────────────────────────────────────────
            ->filters([
                SelectFilter::make('status')
                    ->label('Durum')
                    ->options([
                        'active'    => 'Aktif',
                        'suspended' => 'Askıya Alındı',
                        'expired'   => 'Süresi Doldu',
                    ])
                    ->native(false),

                Tables\Filters\Filter::make('expiring_soon')
                    ->label('7 Günde Bitiyor')
                    ->query(fn (Builder $query) => $query
                        ->where('status', 'active')
                        ->whereNotNull('valid_until')
                        ->where('valid_until', '<=', now()->addDays(7))
                    )
                    ->toggle(),

                Tables\Filters\Filter::make('domain_locked')
                    ->label('Domain Kilitli')
                    ->query(fn (Builder $query) => $query->whereNotNull('domain')->where('domain', '!=', ''))
                    ->toggle(),

                TrashedFilter::make(),
            ])

            // ── Aksiyonlar ─────────────────────────────────────────────────────
            ->actions([

                // ── 💬 Mesaj Gönder ──────────────────────────────────────────
                Action::make('send_message')
                    ->label('Mesaj')
                    ->icon('heroicon-m-chat-bubble-left-ellipsis')
                    ->color('info')
                    ->tooltip('Müşteriye WP panelinde mesaj gönder')
                    ->modalHeading('Eklenti Paneli Mesajı')
                    ->modalDescription('Bu mesaj müşterinin WordPress yönetici panelinde anlık görünür.')
                    ->modalIcon('heroicon-o-chat-bubble-left-ellipsis')
                    ->form([
                        Textarea::make('admin_message')
                            ->label('Mesaj')
                            ->placeholder('Müşteriye gösterilecek mesajı yazın...')
                            ->rows(4)
                            ->required()
                            ->maxLength(1000)
                            ->default(fn (License $record) => $record->admin_message),
                    ])
                    ->action(function (License $record, array $data): void {
                        $record->update(['admin_message' => $data['admin_message']]);

                        Notification::make()
                            ->title('Mesaj gönderildi!')
                            ->body("Mesaj \"{$record->license_key}\" lisansına kaydedildi.")
                            ->success()
                            ->send();
                    })
                    ->modalSubmitActionLabel('Mesajı Kaydet')
                    ->modalCancelActionLabel('İptal'),

                // ── 🚫 Lisansı İptal Et ───────────────────────────────────────
                Action::make('suspend')
                    ->label('İptal Et')
                    ->icon('heroicon-m-no-symbol')
                    ->color('danger')
                    ->tooltip('Lisansı askıya al (Kill-Switch)')
                    ->visible(fn (License $record) => $record->status !== 'suspended')
                    ->requiresConfirmation()
                    ->modalHeading('Lisansı Askıya Al')
                    ->modalDescription(
                        fn (License $record) =>
                            "\"{$record->license_key}\" ({$record->customer_name}) lisansını askıya almak istediğinize emin misiniz? "
                            . "Müşteri bir sonraki doğrulamada erişimini kaybedecektir."
                    )
                    ->modalIcon('heroicon-o-exclamation-triangle')
                    ->modalIconColor('danger')
                    ->modalSubmitActionLabel('Evet, Askıya Al')
                    ->modalCancelActionLabel('İptal')
                    ->action(function (License $record): void {
                        $record->update(['status' => 'suspended']);

                        Notification::make()
                            ->title('Lisans askıya alındı!')
                            ->body("{$record->license_key} artık aktif değil.")
                            ->danger()
                            ->send();
                    }),

                // ── ✅ Lisansı Aktive Et (iptal edilmiş olanlar için) ─────────
                Action::make('activate')
                    ->label('Aktive Et')
                    ->icon('heroicon-m-check-circle')
                    ->color('success')
                    ->tooltip('Askıya alınan lisansı yeniden aktive et')
                    ->visible(fn (License $record) => $record->status === 'suspended')
                    ->requiresConfirmation()
                    ->modalHeading('Lisansı Aktive Et')
                    ->modalDescription(fn (License $record) =>
                        "\"{$record->license_key}\" lisansını yeniden aktive etmek istediğinize emin misiniz?"
                    )
                    ->modalSubmitActionLabel('Evet, Aktive Et')
                    ->action(function (License $record): void {
                        $record->update(['status' => 'active']);

                        Notification::make()
                            ->title('Lisans aktive edildi!')
                            ->body("{$record->license_key} artık aktif.")
                            ->success()
                            ->send();
                    }),

                // ── 🔍 Manuel Sorgula ─────────────────────────────────────────
                Action::make('inspect')
                    ->label('Sorgula')
                    ->icon('heroicon-m-magnifying-glass')
                    ->color('gray')
                    ->tooltip('Son sorgu bilgilerini görüntüle')
                    ->modalHeading(fn (License $record) => "Lisans Detayı: {$record->license_key}")
                    ->modalIcon('heroicon-o-information-circle')
                    ->modalIconColor('info')
                    ->modalContent(function (License $record) {
                        $lastLog = LicenseActivationLog::where('license_id', $record->id)
                            ->whereIn('action', ['verified', 'activated'])
                            ->latest('created_at')
                            ->first();

                        return view('filament.license-inspect-modal', [
                            'license' => $record,
                            'lastLog' => $lastLog,
                        ]);
                    })
                    ->modalSubmitAction(false)
                    ->modalCancelActionLabel('Kapat'),

                ActionGroup::make([
                    Tables\Actions\EditAction::make()->label('Düzenle'),
                    Tables\Actions\DeleteAction::make()->label('Sil'),
                    Tables\Actions\RestoreAction::make()->label('Geri Al'),
                ])->label('Diğer')->icon('heroicon-m-ellipsis-vertical'),
            ])

            ->bulkActions([
                Tables\Actions\BulkActionGroup::make([
                    Tables\Actions\DeleteBulkAction::make()->label('Seçilenleri Sil'),
                    Tables\Actions\RestoreBulkAction::make()->label('Seçilenleri Geri Al'),

                    Tables\Actions\BulkAction::make('bulk_suspend')
                        ->label('Seçilenleri Askıya Al')
                        ->icon('heroicon-m-no-symbol')
                        ->color('danger')
                        ->requiresConfirmation()
                        ->action(fn ($records) => $records->each->update(['status' => 'suspended']))
                        ->deselectRecordsAfterCompletion(),
                ]),
            ])

            ->defaultSort('created_at', 'desc')
            ->striped()
            ->paginated([10, 25, 50, 100])
            ->poll('60s');
    }

    // =========================================================================
    // PAGES
    // =========================================================================

    public static function getRelations(): array
    {
        return [];
    }

    public static function getPages(): array
    {
        return [
            'index'  => Pages\ListLicenses::route('/'),
            'create' => Pages\CreateLicense::route('/create'),
            'edit'   => Pages\EditLicense::route('/{record}/edit'),
        ];
    }

    public static function getEloquentQuery(): Builder
    {
        return parent::getEloquentQuery()
            ->withoutGlobalScopes([SoftDeletingScope::class]);
    }

    public static function getNavigationBadge(): ?string
    {
        // Kritik lisans sayısını nav badge olarak göster
        $critical = License::where(function ($q) {
            $q->where('status', 'expired')
              ->orWhere(function ($inner) {
                  $inner->where('status', 'active')
                        ->whereNotNull('valid_until')
                        ->where('valid_until', '<=', now()->addDays(7));
              });
        })->count();

        return $critical > 0 ? (string) $critical : null;
    }

    public static function getNavigationBadgeColor(): ?string
    {
        return 'danger';
    }
}
