<?php

namespace App\Http\Middleware;

use App\Models\License;
use Closure;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Cache;
use Symfony\Component\HttpFoundation\Response;

/**
 * HMAC-SHA256 tabanlı API kimlik doğrulama middleware'i.
 *
 * Zorunlu headers:
 *   X-License-Key  — Lisans anahtarı (secret_token'ı çözmek için)
 *   X-Signature    — HMAC-SHA256 imzası (hex)
 *   X-Timestamp    — Unix timestamp (saniye cinsinden)
 *   X-Nonce        — Tek kullanımlık rastgele değer (min 16 karakter)
 *
 * İmza Oluşturma (istemci tarafı):
 *   $payload = $method . "\n" . $path . "\n" . $timestamp . "\n" . $nonce . "\n" . $body;
 *   $signature = hash_hmac('sha256', $payload, $secretToken);
 *
 * Güvenlik katmanları:
 *   1. Zorunlu header varlık kontrolü
 *   2. Timestamp penceresi kontrolü (±5 dk) → replay attack koruması
 *   3. Nonce tekrar kullanım kontrolü (Cache ile) → replay attack koruması
 *   4. HMAC imza doğrulaması (timing-safe comparison)
 */
class VerifyHmacSignature
{
    /** İzin verilen maksimum zaman farkı (saniye) */
    private const MAX_TIMESTAMP_DRIFT_SECONDS = 300; // 5 dakika

    /** Nonce cache TTL — zaman penceresinin 2 katı (güvenli tampon) */
    private const NONCE_CACHE_TTL_SECONDS = self::MAX_TIMESTAMP_DRIFT_SECONDS * 2;

    /** Minimum nonce uzunluğu */
    private const MIN_NONCE_LENGTH = 16;

    public function handle(Request $request, Closure $next): Response
    {
        // 1. Zorunlu header'ları kontrol et
        $missingHeaders = $this->getMissingHeaders($request);
        if (!empty($missingHeaders)) {
            return $this->unauthorized(
                'Eksik zorunlu header(lar): ' . implode(', ', $missingHeaders),
                'MISSING_HEADERS'
            );
        }

        $licenseKey = $request->header('X-License-Key');
        $signature  = $request->header('X-Signature');
        $timestamp  = $request->header('X-Timestamp');
        $nonce      = $request->header('X-Nonce');

        // 2. Timestamp format kontrolü
        if (!ctype_digit((string) $timestamp)) {
            return $this->unauthorized('Geçersiz timestamp formatı.', 'INVALID_TIMESTAMP');
        }

        // 3. Timestamp penceresi kontrolü (replay attack önleme)
        $timestampDrift = abs(time() - (int) $timestamp);
        if ($timestampDrift > self::MAX_TIMESTAMP_DRIFT_SECONDS) {
            return $this->unauthorized(
                'İstek zaman aşımına uğradı. Lütfen sisteminizin saatini kontrol edin.',
                'TIMESTAMP_EXPIRED'
            );
        }

        // 4. Nonce minimum uzunluk kontrolü
        if (strlen($nonce) < self::MIN_NONCE_LENGTH) {
            return $this->unauthorized(
                'Nonce en az ' . self::MIN_NONCE_LENGTH . ' karakter olmalıdır.',
                'INVALID_NONCE'
            );
        }

        // 5. Nonce tekrar kullanım kontrolü
        $nonceCacheKey = 'hmac_nonce:' . hash('sha256', $licenseKey . ':' . $nonce);
        if (Cache::has($nonceCacheKey)) {
            return $this->unauthorized(
                'Bu nonce daha önce kullanılmış. Replay attack tespit edildi.',
                'NONCE_REPLAY'
            );
        }

        // 6. Lisans kaydını ve secret_token'ı al
        $license = License::where('license_key', $licenseKey)->first();
        if (!$license) {
            // Timing-safe: lisans bulunamasa bile sabit süre harca
            hash_hmac('sha256', 'dummy', random_bytes(32));
            return $this->unauthorized('Geçersiz lisans anahtarı.', 'INVALID_LICENSE');
        }

        // 7. HMAC imzasını doğrula
        $payload       = $this->buildPayload($request, $timestamp, $nonce);
        $expectedSig   = hash_hmac('sha256', $payload, $license->secret_token);

        if (!hash_equals($expectedSig, strtolower($signature))) {
            return $this->unauthorized('İmza doğrulaması başarısız.', 'INVALID_SIGNATURE');
        }

        // 8. Nonce'u cache'e kaydet (zaman penceresi boyunca geçerli)
        Cache::put($nonceCacheKey, true, self::NONCE_CACHE_TTL_SECONDS);

        // 9. Lisans nesnesini request'e ekle (controller'larda kullanım için)
        $request->attributes->set('license', $license);

        return $next($request);
    }

    // ─── Private Helpers ───────────────────────────────────────────────────────

    private function getMissingHeaders(Request $request): array
    {
        $required = ['X-License-Key', 'X-Signature', 'X-Timestamp', 'X-Nonce'];
        return array_filter($required, fn($h) => !$request->hasHeader($h));
    }

    /**
     * İstemcinin de aynı şekilde oluşturması gereken imza payload'ı.
     * Yöntem + path + timestamp + nonce + ham body birleşimi.
     */
    private function buildPayload(Request $request, string $timestamp, string $nonce): string
    {
        return implode("\n", [
            strtoupper($request->method()),
            $request->path(),
            $timestamp,
            $nonce,
            $request->getContent(), // JSON body veya boş string
        ]);
    }

    private function unauthorized(string $message, string $code): Response
    {
        return response()->json([
            'status'    => 'error',
            'code'      => $code,
            'message'   => $message,
            'data'      => null,
            'timestamp' => now()->toIso8601String(),
        ], Response::HTTP_UNAUTHORIZED);
    }
}
