<?php

namespace App\Services;

use App\Models\License;
use RuntimeException;

/**
 * Kriptografik olarak güvenli lisans anahtarı üretici.
 *
 * Format : XXXX-XXXX-XXXX-XXXX
 * Alfabe : 0-9, A-Z  (O ve I harfleri hariç — karışıklığı önlemek için)
 * Entropi : 4 grup × 4 karakter × log2(34) ≈ 78 bit
 */
class LicenseKeyGenerator
{
    /**
     * Karışıklığa yol açan O (sıfır-o) ve I (büyük-i) karakterleri çıkarıldı.
     */
    private const ALPHABET = '0123456789ABCDEFGHJKLMNPQRSTUVWXYZ';

    private const GROUP_LENGTH = 4;
    private const GROUP_COUNT  = 4;
    private const SEPARATOR    = '-';

    /**
     * Çakışmasız, benzersiz bir lisans anahtarı üretir.
     *
     * @param  int  $maxAttempts Maksimum deneme sayısı (sonsuz döngü koruması)
     * @throws RuntimeException  Benzersiz anahtar üretilemezse
     */
    public function generateUnique(int $maxAttempts = 10): string
    {
        for ($i = 0; $i < $maxAttempts; $i++) {
            $key = $this->generate();

            if (!License::where('license_key', $key)->exists()) {
                return $key;
            }
        }

        throw new RuntimeException(
            "Benzersiz lisans anahtarı {$maxAttempts} denemede üretilemedi."
        );
    }

    /**
     * Tek bir lisans anahtarı üretir (benzersizlik garantisi yok).
     *
     * random_bytes() → kriptografik güçlü rastgelelik (CSPRNG)
     */
    public function generate(): string
    {
        $groups = [];

        for ($g = 0; $g < self::GROUP_COUNT; $g++) {
            $groups[] = $this->randomGroup(self::GROUP_LENGTH);
        }

        return implode(self::SEPARATOR, $groups);
    }

    /**
     * HMAC imzalama için kriptografik rastgele secret token üretir.
     * 32 byte → 64 karakter hex string
     */
    public function generateSecretToken(): string
    {
        return bin2hex(random_bytes(32));
    }

    // ─── Private Helpers ───────────────────────────────────────────────────────

    /**
     * random_bytes() ile alfabe üzerinden önyargısız rastgele grup üretir.
     * Modulo bias'ı önlemek için rejection sampling uygulanır.
     */
    private function randomGroup(int $length): string
    {
        $alphabetLength = strlen(self::ALPHABET);
        // En yakın 2^n sınırına tavan alma — rejection sampling eşiği
        $mask   = (int) (2 ** ceil(log($alphabetLength, 2))) - 1;
        $result = '';

        while (strlen($result) < $length) {
            // 1 byte al, maske uygula; alfabeyi aşıyorsa at (bias'sız)
            $byte = ord(random_bytes(1)) & $mask;

            if ($byte < $alphabetLength) {
                $result .= self::ALPHABET[$byte];
            }
        }

        return $result;
    }
}
